/*
 * Copyright 2024 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package androidx.compose.animation

import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableFloatStateOf
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.geometry.Offset
import androidx.compose.ui.graphics.drawscope.ContentDrawScope
import androidx.compose.ui.graphics.drawscope.DrawScope
import androidx.compose.ui.graphics.drawscope.translate
import androidx.compose.ui.graphics.layer.GraphicsLayer
import androidx.compose.ui.graphics.layer.drawLayer
import androidx.compose.ui.layout.Measurable
import androidx.compose.ui.layout.MeasureResult
import androidx.compose.ui.layout.MeasureScope
import androidx.compose.ui.modifier.ModifierLocalModifierNode
import androidx.compose.ui.node.DrawModifierNode
import androidx.compose.ui.node.LayoutModifierNode
import androidx.compose.ui.node.ModifierNodeElement
import androidx.compose.ui.node.invalidateDraw
import androidx.compose.ui.node.requireGraphicsContext
import androidx.compose.ui.node.requireLayoutCoordinates
import androidx.compose.ui.platform.InspectorInfo
import androidx.compose.ui.unit.Constraints

internal class RenderInTransitionOverlayNodeElement(
    var sharedTransitionScope: SharedTransitionScopeImpl,
    var renderInOverlay: () -> Boolean,
    val zIndexInOverlay: Float,
) : ModifierNodeElement<RenderInTransitionOverlayNode>() {
    override fun create(): RenderInTransitionOverlayNode {
        return RenderInTransitionOverlayNode(
            sharedTransitionScope,
            renderInOverlay,
            zIndexInOverlay,
        )
    }

    override fun update(node: RenderInTransitionOverlayNode) {
        node.sharedScope = sharedTransitionScope
        node.renderInOverlay = renderInOverlay
        node.zIndexInOverlay = zIndexInOverlay
    }

    override fun hashCode(): Int =
        (sharedTransitionScope.hashCode() * 31 + renderInOverlay.hashCode()) * 31 +
            zIndexInOverlay.hashCode()

    override fun equals(other: Any?): Boolean {
        if (other is RenderInTransitionOverlayNodeElement) {
            return sharedTransitionScope == other.sharedTransitionScope &&
                renderInOverlay === other.renderInOverlay &&
                zIndexInOverlay == other.zIndexInOverlay
        }
        return false
    }

    override fun InspectorInfo.inspectableProperties() {
        name = "renderInSharedTransitionOverlay"
        properties["sharedTransitionScope"] = sharedTransitionScope
        properties["renderInOverlay"] = renderInOverlay
        properties["zIndexInOverlay"] = zIndexInOverlay
    }
}

internal class RenderInTransitionOverlayNode(
    var sharedScope: SharedTransitionScopeImpl,
    var renderInOverlay: () -> Boolean,
    zIndexInOverlay: Float,
) : Modifier.Node(), LayoutModifierNode, DrawModifierNode, ModifierLocalModifierNode {
    var zIndexInOverlay by mutableFloatStateOf(zIndexInOverlay)

    val parentState: SharedElementEntry?
        get() = ModifierLocalSharedElementInternalState.current

    /**
     * Both enabled and positionInOverlay below are mutated during approach placement. Once they are
     * updated during the layout phase, they will explicitly trigger invalidation of drawing for
     * both the SharedTransition overlay as well as the RenderInTransitionOverlayNode. In other
     * words, the state observation only needs to happen during placement, the drawing stage will be
     * invalidated **in the same frame** as needed. Therefore, both the enabled and
     * positionInOverlay properties are intentionally defined as non-snapshot states, as they are
     * not meant to be observed.
     */
    private var enabled: Boolean = false
        set(value) {
            if (value != field) {
                this@RenderInTransitionOverlayNode.sharedScope.invalidateOverlay?.invoke()
                invalidateDraw()
                field = value
            }
        }

    private var positionInOverlay: Offset = Offset.Zero
        set(value) {
            if (value != field) {
                this@RenderInTransitionOverlayNode.sharedScope.invalidateOverlay?.invoke()
                invalidateDraw()
                field = value
            }
        }

    private inner class LayerWithRenderer(val layer: GraphicsLayer) : LayerRenderer {
        override val parentState: SharedElementEntry?
            get() = this@RenderInTransitionOverlayNode.parentState

        override val zIndex: Float
            get() = this@RenderInTransitionOverlayNode.zIndexInOverlay

        override fun drawInOverlay(drawScope: DrawScope) {
            if (enabled) {
                with(drawScope) {
                    translate(positionInOverlay.x, positionInOverlay.y) { drawLayer(layer) }
                }
            }
        }
    }

    override fun MeasureScope.measure(
        measurable: Measurable,
        constraints: Constraints,
    ): MeasureResult {
        return measurable.measure(constraints).run {
            layout(width, height) {
                if (!isLookingAhead) {
                    if (renderInOverlay()) {
                        // Access coordinates from the PlacementScope to ensure position changes
                        // while `renderInOverlay` is enabled triggers changes in the position
                        // in overlay, and invalidations in drawing.
                        coordinates?.let {
                            enabled = true
                            positionInOverlay =
                                sharedScope.root.localPositionOf(
                                    this@RenderInTransitionOverlayNode.requireLayoutCoordinates(),
                                    Offset.Zero,
                                )
                        }
                    } else {
                        enabled = false
                    }
                }
                place(0, 0)
            }
        }
    }

    // Render in-place logic. Depending on the result of `renderInOverlay()`, the content will
    // either render in-place or in the overlay, but never in both places.
    override fun ContentDrawScope.draw() {
        val layer = requireNotNull(layer) { "Error: layer never initialized" }
        layer.record { this@draw.drawContent() }
        if (!enabled) {
            drawLayer(layer)
        }
    }

    val layer: GraphicsLayer?
        get() = layerWithRenderer?.layer

    private var layerWithRenderer: LayerWithRenderer? = null

    override fun onAttach() {
        LayerWithRenderer(requireGraphicsContext().createGraphicsLayer()).let {
            sharedScope.onLayerRendererCreated(it)
            layerWithRenderer = it
        }
    }

    override fun onDetach() {
        layerWithRenderer?.let {
            sharedScope.onLayerRendererRemoved(it)
            requireGraphicsContext().releaseGraphicsLayer(it.layer)
        }
    }
}
