/*
 * Copyright 2024 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package androidx.compose.material3

import androidx.compose.runtime.Composable
import androidx.compose.runtime.ReadOnlyComposable
import kotlinx.cinterop.ExperimentalForeignApi
import kotlinx.cinterop.toKString
import platform.posix.getenv

/**
 * A simple locale class for Linux based on language tag.
 */
actual class CalendarLocale(val languageTag: String) {
    val language: String
        get() = languageTag.substringBefore('_').substringBefore('-')
    
    val region: String
        get() {
            val afterLang = languageTag.substringAfter('_', "").substringAfter('-', "")
            return afterLang.substringBefore('.').substringBefore('@').uppercase()
        }
    
    override fun equals(other: Any?): Boolean {
        if (this === other) return true
        if (other !is CalendarLocale) return false
        return languageTag == other.languageTag
    }
    
    override fun hashCode(): Int = languageTag.hashCode()
    
    override fun toString(): String = languageTag
}

@OptIn(ExperimentalForeignApi::class)
@Composable
@ReadOnlyComposable
internal actual fun defaultLocale(): CalendarLocale {
    val localeEnv = getenv("LC_TIME")?.toKString()
        ?: getenv("LC_ALL")?.toKString()
        ?: getenv("LANG")?.toKString()
        ?: "en_US"
    return CalendarLocale(localeEnv)
}
