/*
 * Copyright 2024 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

@file:OptIn(ExperimentalTime::class)

package androidx.compose.material3.internal

import androidx.compose.material3.CalendarLocale
import kotlin.time.ExperimentalTime
import kotlinx.datetime.Instant
import kotlinx.datetime.LocalDate
import kotlinx.datetime.TimeZone
import kotlinx.datetime.atStartOfDayIn
import kotlinx.datetime.toLocalDateTime

internal actual class PlatformDateFormat actual constructor(private val locale: CalendarLocale) {

    actual val firstDayOfWeek: Int
        get() = getFirstDayOfWeekForLocale()

    actual fun formatWithPattern(
        utcTimeMillis: Long,
        pattern: String,
        cache: MutableMap<String, Any>
    ): String {
        val instant = Instant.fromEpochMilliseconds(utcTimeMillis)
        val localDateTime = instant.toLocalDateTime(TimeZone.UTC)
        
        return formatDateTimeWithPattern(localDateTime, pattern)
    }

    actual fun formatWithSkeleton(
        utcTimeMillis: Long,
        skeleton: String,
        cache: MutableMap<String, Any>
    ): String {
        // Convert skeleton to a simple pattern
        val pattern = skeletonToPattern(skeleton)
        return formatWithPattern(utcTimeMillis, pattern, cache)
    }

    actual fun parse(
        date: String,
        pattern: String,
        locale: CalendarLocale,
        cache: MutableMap<String, Any>
    ): CalendarDate? {
        return try {
            parseDateWithPattern(date, pattern)
        } catch (e: Exception) {
            null
        }
    }

    actual fun getDateInputFormat(): DateInputFormat {
        // Return a sensible default based on locale
        val pattern = when (locale.language) {
            "en" -> if (locale.region == "US") "MM/dd/yyyy" else "dd/MM/yyyy"
            "de", "ru", "fr", "es", "it", "pt" -> "dd/MM/yyyy"
            "ja", "zh", "ko" -> "yyyy/MM/dd"
            else -> "dd/MM/yyyy"
        }
        return datePatternAsInputFormat(pattern)
    }

    actual val weekdayNames: List<Pair<String, String>>
        get() = listOf(
            "Monday" to "Mo",
            "Tuesday" to "Tu",
            "Wednesday" to "We",
            "Thursday" to "Th",
            "Friday" to "Fr",
            "Saturday" to "Sa",
            "Sunday" to "Su"
        )

    actual fun is24HourFormat(): Boolean {
        // Most locales use 24-hour format except US/UK
        return when (locale.language) {
            "en" -> locale.region !in listOf("US", "")
            else -> true
        }
    }

    private fun getFirstDayOfWeekForLocale(): Int {
        // ISO weekday: Monday = 1, Sunday = 7
        // Most of the world uses Monday as first day
        // US, Canada, Japan use Sunday
        return when {
            locale.region in listOf("US", "CA", "JP") -> 7 // Sunday
            else -> 1 // Monday
        }
    }

    private fun formatDateTimeWithPattern(
        dateTime: kotlinx.datetime.LocalDateTime,
        pattern: String
    ): String {
        var result = pattern
        
        // Year
        result = result.replace("yyyy", dateTime.year.toString().padStart(4, '0'))
        result = result.replace("yy", (dateTime.year % 100).toString().padStart(2, '0'))
        
        // Month
        result = result.replace("MMMM", getMonthName(dateTime.monthNumber, full = true))
        result = result.replace("MMM", getMonthName(dateTime.monthNumber, full = false))
        result = result.replace("MM", dateTime.monthNumber.toString().padStart(2, '0'))
        result = result.replace(Regex("(?<!M)M(?!M)"), dateTime.monthNumber.toString())
        
        // Day
        result = result.replace("dd", dateTime.dayOfMonth.toString().padStart(2, '0'))
        result = result.replace(Regex("(?<!d)d(?!d)"), dateTime.dayOfMonth.toString())
        
        // Day of week
        result = result.replace("EEEE", getDayOfWeekName(dateTime.dayOfWeek.ordinal + 1, full = true))
        result = result.replace("EEE", getDayOfWeekName(dateTime.dayOfWeek.ordinal + 1, full = false))
        result = result.replace("EE", getDayOfWeekName(dateTime.dayOfWeek.ordinal + 1, full = false))
        result = result.replace(Regex("(?<!E)E(?!E)"), getDayOfWeekName(dateTime.dayOfWeek.ordinal + 1, full = false))
        
        // Hour
        val hour24 = dateTime.hour
        val hour12 = if (hour24 == 0) 12 else if (hour24 > 12) hour24 - 12 else hour24
        result = result.replace("HH", hour24.toString().padStart(2, '0'))
        result = result.replace(Regex("(?<!H)H(?!H)"), hour24.toString())
        result = result.replace("hh", hour12.toString().padStart(2, '0'))
        result = result.replace(Regex("(?<!h)h(?!h)"), hour12.toString())
        
        // Minute
        result = result.replace("mm", dateTime.minute.toString().padStart(2, '0'))
        result = result.replace(Regex("(?<!m)m(?!m)"), dateTime.minute.toString())
        
        // Second
        result = result.replace("ss", dateTime.second.toString().padStart(2, '0'))
        result = result.replace(Regex("(?<!s)s(?!s)"), dateTime.second.toString())
        
        // AM/PM
        val amPm = if (hour24 < 12) "AM" else "PM"
        result = result.replace("a", amPm)
        
        return result
    }

    private fun skeletonToPattern(skeleton: String): String {
        // Simple skeleton to pattern conversion
        var pattern = skeleton
        
        // Handle common skeletons
        if (skeleton.contains("yMMM") && !skeleton.contains("d")) {
            pattern = "MMM yyyy"
        } else if (skeleton.contains("yMMMM")) {
            pattern = skeleton.replace("yMMMM", "MMMM yyyy")
        } else if (skeleton.contains("yMMM")) {
            pattern = skeleton.replace("yMMM", "MMM yyyy")
        }
        
        return pattern
    }

    private fun parseDateWithPattern(date: String, pattern: String): CalendarDate? {
        // Simple date parsing - extract year, month, day based on pattern
        val yearIndex = pattern.indexOf("yyyy").takeIf { it >= 0 } ?: pattern.indexOf("yy")
        val monthIndex = pattern.indexOf("MM")
        val dayIndex = pattern.indexOf("dd")

        if (yearIndex < 0 || monthIndex < 0 || dayIndex < 0) return null

        try {
            // Find the delimiters used
            val delimiter = pattern.firstOrNull { !it.isLetter() } ?: '/'
            val parts = date.split(delimiter)
            if (parts.size < 3) return null

            // Determine order based on pattern
            data class DatePart(val type: Char, val index: Int)
            val order = listOf(
                DatePart('y', yearIndex),
                DatePart('M', monthIndex),
                DatePart('d', dayIndex)
            ).sortedBy { it.index }

            var year = 0
            var month = 0
            var day = 0

            order.forEachIndexed { i, part ->
                val value = parts.getOrNull(i)?.toIntOrNull() ?: return null
                when (part.type) {
                    'y' -> year = if (value < 100) 2000 + value else value
                    'M' -> month = value
                    'd' -> day = value
                }
            }

            if (year < 1 || month < 1 || month > 12 || day < 1 || day > 31) return null

            // Calculate UTC millis for this date
            val localDate = LocalDate(year, month, day)
            val utcMillis = localDate.atStartOfDayIn(TimeZone.UTC).toEpochMilliseconds()
            
            return CalendarDate(year, month, day, utcMillis)
        } catch (e: Exception) {
            return null
        }
    }

    private fun getMonthName(month: Int, full: Boolean): String {
        val names = if (full) {
            listOf("January", "February", "March", "April", "May", "June",
                   "July", "August", "September", "October", "November", "December")
        } else {
            listOf("Jan", "Feb", "Mar", "Apr", "May", "Jun",
                   "Jul", "Aug", "Sep", "Oct", "Nov", "Dec")
        }
        return names.getOrElse(month - 1) { "" }
    }

    private fun getDayOfWeekName(dayOfWeek: Int, full: Boolean): String {
        val names = if (full) {
            listOf("Monday", "Tuesday", "Wednesday", "Thursday", "Friday", "Saturday", "Sunday")
        } else {
            listOf("Mon", "Tue", "Wed", "Thu", "Fri", "Sat", "Sun")
        }
        return names.getOrElse(dayOfWeek - 1) { "" }
    }
}
