/*
 * Copyright 2024 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package androidx.compose.ui.platform

import androidx.compose.ui.InternalComposeUiApi
import androidx.compose.ui.geometry.Offset
import androidx.compose.ui.input.pointer.PointerIcon

/**
 * Linux-specific implementation of [PlatformContext].
 * 
 * This provides the platform bindings needed for Compose UI to work on native Linux.
 * It extends [PlatformContext.Empty] which provides sensible defaults for most functionality.
 */
@InternalComposeUiApi
class LinuxPlatformContext(
    /**
     * Optional callback to set the pointer/cursor icon.
     * The callback receives the [PointerIcon] that should be displayed.
     */
    private val onSetPointerIcon: ((PointerIcon) -> Unit)? = null,
    
    /**
     * Optional callback to convert local coordinates to window coordinates.
     * This is useful when the Compose scene is offset within the window.
     */
    private val localToWindowConverter: ((Offset) -> Offset)? = null,
    
    /**
     * Optional callback to convert window coordinates to local coordinates.
     */
    private val windowToLocalConverter: ((Offset) -> Offset)? = null,
    
    /**
     * Optional callback to convert local coordinates to screen coordinates.
     */
    private val localToScreenConverter: ((Offset) -> Offset)? = null,
    
    /**
     * Optional callback to convert screen coordinates to local coordinates.
     */
    private val screenToLocalConverter: ((Offset) -> Offset)? = null,
    
    /**
     * Whether the window is transparent.
     */
    private val windowTransparent: Boolean = false
) : PlatformContext.Empty() {
    
    override val isWindowTransparent: Boolean
        get() = windowTransparent
    
    override fun setPointerIcon(pointerIcon: PointerIcon) {
        onSetPointerIcon?.invoke(pointerIcon)
    }
    
    override fun convertLocalToWindowPosition(localPosition: Offset): Offset {
        return localToWindowConverter?.invoke(localPosition) ?: localPosition
    }
    
    override fun convertWindowToLocalPosition(positionInWindow: Offset): Offset {
        return windowToLocalConverter?.invoke(positionInWindow) ?: positionInWindow
    }
    
    override fun convertLocalToScreenPosition(localPosition: Offset): Offset {
        return localToScreenConverter?.invoke(localPosition) 
            ?: convertLocalToWindowPosition(localPosition)
    }
    
    override fun convertScreenToLocalPosition(positionOnScreen: Offset): Offset {
        return screenToLocalConverter?.invoke(positionOnScreen)
            ?: convertWindowToLocalPosition(positionOnScreen)
    }
}
