/*
 * Copyright 2025 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package androidx.compose.ui.platform

import androidx.compose.ui.ExperimentalComposeUiApi

/**
 * On Linux, the native clipboard is represented by a simple string holder.
 * This can be replaced with krwin-toolkit's Wayland clipboard integration.
 */
actual typealias NativeClipboard = LinuxClipboardHolder

/**
 * A simple clipboard holder for Linux.
 * TODO: Replace with krwin-toolkit's smithay-clipboard integration for Wayland.
 */
class LinuxClipboardHolder {
    internal var content: String? = null
}

internal class LinuxPlatformClipboard : Clipboard {
    private val clipboardHolder = LinuxClipboardHolder()

    override suspend fun getClipEntry(): ClipEntry? {
        val text = clipboardHolder.content ?: return null
        return ClipEntry.withPlainText(text)
    }

    override suspend fun setClipEntry(clipEntry: ClipEntry?) {
        clipboardHolder.content = clipEntry?.plainText
    }

    override val nativeClipboard: NativeClipboard
        get() = clipboardHolder
}

internal actual fun createPlatformClipboard(): Clipboard {
    return LinuxPlatformClipboard()
}

/**
 * A wrapper for Linux clipboard items.
 * Currently, it operates only with plain text strings.
 * TODO: Wire up to krwin-toolkit's Wayland clipboard for proper system integration.
 */
actual class ClipEntry internal constructor() {

    // TODO https://youtrack.jetbrains.com/issue/CMP-1260/ClipboardManager.-Implement-getClip-getClipMetadata-setClip
    actual val clipMetadata: ClipMetadata
        get() = TODO("ClipMetadata is not implemented for Linux")

    internal var plainText: String? = null

    @ExperimentalComposeUiApi
    fun getPlainText(): String? = plainText

    companion object {
        @ExperimentalComposeUiApi
        fun withPlainText(text: String): ClipEntry = ClipEntry().apply {
            plainText = text
        }
    }
}
