/*
 * Copyright 2024 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package androidx.compose.ui.window

import androidx.compose.runtime.Composable
import androidx.compose.ui.InternalComposeUiApi
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.graphics.asComposeCanvas
import androidx.compose.ui.input.pointer.PointerButton
import androidx.compose.ui.input.pointer.PointerButtons
import androidx.compose.ui.input.pointer.PointerEventType
import androidx.compose.ui.input.pointer.PointerId
import androidx.compose.ui.input.pointer.PointerKeyboardModifiers
import androidx.compose.ui.input.pointer.PointerType
import androidx.compose.ui.platform.LinuxPlatformContext
import androidx.compose.ui.scene.ComposeSceneContext
import androidx.compose.ui.scene.PlatformLayersComposeScene
import androidx.compose.ui.unit.Density
import androidx.compose.ui.unit.IntSize
import androidx.compose.ui.unit.LayoutDirection
import kotlinx.coroutines.Dispatchers
import org.jetbrains.skia.*
import kotlin.coroutines.CoroutineContext

/**
 * Configuration for [ComposeWindow].
 */
class ComposeWindowConfiguration {
    /**
     * Initial window width in pixels.
     */
    var width: Int = 800
    
    /**
     * Initial window height in pixels.
     */
    var height: Int = 600
    
    /**
     * Window title.
     */
    var title: String = "Compose Window"
    
    /**
     * Background color for the window.
     */
    var backgroundColor: Color = Color(0xFF1E1E2E)
}

/**
 * Represents a native Linux window that hosts Compose UI content.
 * 
 * This class provides the integration between the native windowing system (via krwin-toolkit)
 * and Compose Multiplatform rendering (via Skiko).
 * 
 * Usage:
 * ```
 * val window = ComposeWindow(configure = {
 *     width = 800
 *     height = 600
 *     title = "My Compose App"
 * }) {
 *     MaterialTheme {
 *         Text("Hello, Linux!")
 *     }
 * }
 * window.run()
 * ```
 */
@OptIn(InternalComposeUiApi::class)
class ComposeWindow(
    configure: ComposeWindowConfiguration.() -> Unit = {},
    private val content: @Composable () -> Unit
) {
    private val configuration = ComposeWindowConfiguration().apply(configure)
    
    // These will be initialized when the window is created
    private var windowWidth: Int = configuration.width
    private var windowHeight: Int = configuration.height
    private var fbWidth: Int = configuration.width
    private var fbHeight: Int = configuration.height
    private var scaleFactor: Float = 1f
    
    // Pointer state
    private var mouseX: Float = 0f
    private var mouseY: Float = 0f
    private var mousePressed: Boolean = false
    private var pointerId: Long = 0L
    
    // Scene and rendering
    private var scene: androidx.compose.ui.scene.ComposeScene? = null
    private var needsRedraw: Boolean = true
    
    private val density: Density
        get() = Density(scaleFactor)
    
    /**
     * Interface for platform-specific window operations.
     * This allows the window to be used with different windowing backends.
     */
    interface NativeWindowBackend {
        fun create(width: Int, height: Int, title: String): Boolean
        fun makeCurrent()
        fun swapBuffers()
        fun pollEvents(): Boolean
        fun getEvents(): List<Any>
        fun getSize(): WindowSize
        fun cleanup()
        fun getGlProcAddress(): Long
    }
    
    data class WindowSize(
        val width: Int,
        val height: Int,
        val framebufferWidth: Int,
        val framebufferHeight: Int,
        val scaleFactor: Float
    )
    
    /**
     * Pointer event data from the native window.
     */
    data class PointerEvent(
        val x: Float,
        val y: Float,
        val type: PointerEventKind,
        val button: Int = 0
    )
    
    enum class PointerEventKind {
        MOVE, PRESS, RELEASE, ENTER, EXIT
    }
    
    /**
     * Resize event data.
     */
    data class ResizeEvent(
        val width: Int,
        val height: Int,
        val framebufferWidth: Int,
        val framebufferHeight: Int,
        val scaleFactor: Float
    )
    
    private fun createScene(coroutineContext: CoroutineContext): androidx.compose.ui.scene.ComposeScene {
        val platformContext = LinuxPlatformContext()
        
        val sceneContext = object : ComposeSceneContext {
            override val platformContext = platformContext
        }
        
        return PlatformLayersComposeScene(
            density = density,
            layoutDirection = LayoutDirection.Ltr,
            size = IntSize(fbWidth, fbHeight),
            coroutineContext = coroutineContext,
            composeSceneContext = sceneContext,
            invalidate = { needsRedraw = true }
        ).also {
            it.setContent(content)
        }
    }
    
    private fun updateSceneSize() {
        scene?.let { s ->
            s.density = density
            s.size = IntSize(fbWidth, fbHeight)
        }
    }
    
    /**
     * Handles a pointer event from the native window.
     */
    fun handlePointerEvent(event: PointerEvent) {
        mouseX = event.x * scaleFactor
        mouseY = event.y * scaleFactor
        
        val eventType = when (event.type) {
            PointerEventKind.MOVE -> PointerEventType.Move
            PointerEventKind.PRESS -> {
                mousePressed = true
                pointerId++
                PointerEventType.Press
            }
            PointerEventKind.RELEASE -> {
                mousePressed = false
                PointerEventType.Release
            }
            PointerEventKind.ENTER -> PointerEventType.Enter
            PointerEventKind.EXIT -> PointerEventType.Exit
        }
        
        val buttons = PointerButtons(isPrimaryPressed = mousePressed)
        
        scene?.sendPointerEvent(
            eventType = eventType,
            position = androidx.compose.ui.geometry.Offset(mouseX, mouseY),
            buttons = buttons,
            keyboardModifiers = PointerKeyboardModifiers(0),
            nativeEvent = null,
            button = if (event.type == PointerEventKind.PRESS || event.type == PointerEventKind.RELEASE) {
                PointerButton.Primary
            } else null
        )
        
        needsRedraw = true
    }
    
    /**
     * Handles a resize event from the native window.
     */
    fun handleResizeEvent(event: ResizeEvent) {
        windowWidth = event.width
        windowHeight = event.height
        fbWidth = event.framebufferWidth
        fbHeight = event.framebufferHeight
        scaleFactor = event.scaleFactor
        updateSceneSize()
        needsRedraw = true
    }
    
    /**
     * Renders the Compose content to the given Skia canvas.
     */
    fun render(canvas: Canvas, nanoTime: Long) {
        scene?.render(canvas.asComposeCanvas(), nanoTime)
    }
    
    /**
     * Returns true if the scene needs to be redrawn.
     */
    fun needsRedraw(): Boolean = needsRedraw
    
    /**
     * Marks the scene as having been drawn.
     */
    fun markDrawn() {
        needsRedraw = false
    }
    
    /**
     * Initializes the Compose scene with the given coroutine context.
     * Call this after the GL context is made current.
     */
    fun initialize(coroutineContext: CoroutineContext = Dispatchers.Main) {
        scene = createScene(coroutineContext)
    }
    
    /**
     * Updates the scene density and size based on window parameters.
     */
    fun updateFromWindowSize(
        width: Int,
        height: Int,
        framebufferWidth: Int,
        framebufferHeight: Int,
        scale: Float
    ) {
        windowWidth = width
        windowHeight = height
        fbWidth = framebufferWidth
        fbHeight = framebufferHeight
        scaleFactor = scale
        updateSceneSize()
    }
    
    /**
     * Closes the Compose scene and releases resources.
     */
    fun close() {
        scene?.close()
        scene = null
    }
    
    /**
     * Returns the configuration used to create this window.
     */
    fun getConfiguration(): ComposeWindowConfiguration = configuration
}
