/*
 * Copyright 2024 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package androidx.compose.ui.text.intl

import kotlinx.cinterop.ExperimentalForeignApi
import kotlinx.cinterop.toKString
import platform.posix.getenv

/**
 * A simple locale implementation for Linux.
 * Parses POSIX locale format: language[_territory][.codeset][@modifier]
 * or BCP47 format: language[-script][-region]
 */
actual class PlatformLocale(
    internal val languageTag: String
) {
    internal val parsedLanguage: String
    internal val parsedScript: String
    internal val parsedRegion: String

    init {
        val parts = parseLanguageTag(languageTag)
        parsedLanguage = parts.first
        parsedScript = parts.second
        parsedRegion = parts.third
    }

    private fun parseLanguageTag(tag: String): Triple<String, String, String> {
        // Handle POSIX format: en_US.UTF-8 or en_US
        val cleanTag = tag.substringBefore('.').substringBefore('@')
        
        // Try BCP47 format first (uses hyphens): en-US, zh-Hans-CN
        if (cleanTag.contains('-')) {
            val segments = cleanTag.split('-')
            val lang = segments.getOrNull(0) ?: "en"
            
            // Check if second segment is a script (4 letters) or region (2 letters or 3 digits)
            val second = segments.getOrNull(1) ?: ""
            val third = segments.getOrNull(2) ?: ""
            
            return if (second.length == 4 && second.all { it.isLetter() }) {
                // It's a script: en-Latn-US
                Triple(lang, second, third.uppercase())
            } else {
                // It's a region: en-US
                Triple(lang, "", second.uppercase())
            }
        }
        
        // Handle POSIX format (uses underscores): en_US
        if (cleanTag.contains('_')) {
            val segments = cleanTag.split('_')
            return Triple(
                segments.getOrNull(0) ?: "en",
                "",
                segments.getOrNull(1)?.uppercase() ?: ""
            )
        }
        
        // Just a language code
        return Triple(cleanTag.ifEmpty { "en" }, "", "")
    }

    override fun equals(other: Any?): Boolean {
        if (this === other) return true
        if (other !is PlatformLocale) return false
        return languageTag == other.languageTag
    }

    override fun hashCode(): Int = languageTag.hashCode()

    override fun toString(): String = "PlatformLocale($languageTag)"
}

internal actual val PlatformLocale.language: String
    get() = parsedLanguage

internal actual val PlatformLocale.script: String
    get() = parsedScript

internal actual val PlatformLocale.region: String
    get() = parsedRegion.ifEmpty { "US" }

internal actual fun PlatformLocale.getLanguageTag(): String = languageTag

@OptIn(ExperimentalForeignApi::class)
internal actual fun createPlatformLocaleDelegate(): PlatformLocaleDelegate =
    object : PlatformLocaleDelegate {
        override val current: LocaleList
            get() {
                val localeEnv = getenv("LC_ALL")?.toKString()
                    ?: getenv("LC_MESSAGES")?.toKString()
                    ?: getenv("LANG")?.toKString()
                    ?: "en_US"
                
                return LocaleList(listOf(Locale(PlatformLocale(localeEnv))))
            }

        override fun parseLanguageTag(languageTag: String): PlatformLocale {
            return PlatformLocale(languageTag)
        }
    }

// RTL languages list
private val rtlLanguages = setOf(
    "ar", // Arabic
    "arc", // Aramaic
    "ckb", // Central Kurdish
    "dv", // Divehi
    "fa", // Persian
    "ha", // Hausa
    "he", // Hebrew
    "iw", // Hebrew (old code)
    "khw", // Khowar
    "ks", // Kashmiri
    "ku", // Kurdish
    "ps", // Pashto
    "sd", // Sindhi
    "ur", // Urdu
    "uz_AF", // Uzbek (Afghanistan)
    "yi" // Yiddish
)

internal actual fun PlatformLocale.isRtl(): Boolean =
    language.lowercase() in rtlLanguages
